#include <avr/io.h>
#include <avr/cpufunc.h>
#include <avr/pgmspace.h>
#include <stdio.h>
#include "Common.h"
#include "tests.h"
#include "TinyMT.h"

// Calculate CRC32C
const uint32_t crctable[] PROGMEM = {
	0x00000000L, 0xF26B8303L, 0xE13B70F7L, 0x1350F3F4L,
	0xC79A971FL, 0x35F1141CL, 0x26A1E7E8L, 0xD4CA64EBL,
	0x8AD958CFL, 0x78B2DBCCL, 0x6BE22838L, 0x9989AB3BL,
	0x4D43CFD0L, 0xBF284CD3L, 0xAC78BF27L, 0x5E133C24L,
	0x105EC76FL, 0xE235446CL, 0xF165B798L, 0x030E349BL,
	0xD7C45070L, 0x25AFD373L, 0x36FF2087L, 0xC494A384L,
	0x9A879FA0L, 0x68EC1CA3L, 0x7BBCEF57L, 0x89D76C54L,
	0x5D1D08BFL, 0xAF768BBCL, 0xBC267848L, 0x4E4DFB4BL,
	0x20BD8EDEL, 0xD2D60DDDL, 0xC186FE29L, 0x33ED7D2AL,
	0xE72719C1L, 0x154C9AC2L, 0x061C6936L, 0xF477EA35L,
	0xAA64D611L, 0x580F5512L, 0x4B5FA6E6L, 0xB93425E5L,
	0x6DFE410EL, 0x9F95C20DL, 0x8CC531F9L, 0x7EAEB2FAL,
	0x30E349B1L, 0xC288CAB2L, 0xD1D83946L, 0x23B3BA45L,
	0xF779DEAEL, 0x05125DADL, 0x1642AE59L, 0xE4292D5AL,
	0xBA3A117EL, 0x4851927DL, 0x5B016189L, 0xA96AE28AL,
	0x7DA08661L, 0x8FCB0562L, 0x9C9BF696L, 0x6EF07595L,
	0x417B1DBCL, 0xB3109EBFL, 0xA0406D4BL, 0x522BEE48L,
	0x86E18AA3L, 0x748A09A0L, 0x67DAFA54L, 0x95B17957L,
	0xCBA24573L, 0x39C9C670L, 0x2A993584L, 0xD8F2B687L,
	0x0C38D26CL, 0xFE53516FL, 0xED03A29BL, 0x1F682198L,
	0x5125DAD3L, 0xA34E59D0L, 0xB01EAA24L, 0x42752927L,
	0x96BF4DCCL, 0x64D4CECFL, 0x77843D3BL, 0x85EFBE38L,
	0xDBFC821CL, 0x2997011FL, 0x3AC7F2EBL, 0xC8AC71E8L,
	0x1C661503L, 0xEE0D9600L, 0xFD5D65F4L, 0x0F36E6F7L,
	0x61C69362L, 0x93AD1061L, 0x80FDE395L, 0x72966096L,
	0xA65C047DL, 0x5437877EL, 0x4767748AL, 0xB50CF789L,
	0xEB1FCBADL, 0x197448AEL, 0x0A24BB5AL, 0xF84F3859L,
	0x2C855CB2L, 0xDEEEDFB1L, 0xCDBE2C45L, 0x3FD5AF46L,
	0x7198540DL, 0x83F3D70EL, 0x90A324FAL, 0x62C8A7F9L,
	0xB602C312L, 0x44694011L, 0x5739B3E5L, 0xA55230E6L,
	0xFB410CC2L, 0x092A8FC1L, 0x1A7A7C35L, 0xE811FF36L,
	0x3CDB9BDDL, 0xCEB018DEL, 0xDDE0EB2AL, 0x2F8B6829L,
	0x82F63B78L, 0x709DB87BL, 0x63CD4B8FL, 0x91A6C88CL,
	0x456CAC67L, 0xB7072F64L, 0xA457DC90L, 0x563C5F93L,
	0x082F63B7L, 0xFA44E0B4L, 0xE9141340L, 0x1B7F9043L,
	0xCFB5F4A8L, 0x3DDE77ABL, 0x2E8E845FL, 0xDCE5075CL,
	0x92A8FC17L, 0x60C37F14L, 0x73938CE0L, 0x81F80FE3L,
	0x55326B08L, 0xA759E80BL, 0xB4091BFFL, 0x466298FCL,
	0x1871A4D8L, 0xEA1A27DBL, 0xF94AD42FL, 0x0B21572CL,
	0xDFEB33C7L, 0x2D80B0C4L, 0x3ED04330L, 0xCCBBC033L,
	0xA24BB5A6L, 0x502036A5L, 0x4370C551L, 0xB11B4652L,
	0x65D122B9L, 0x97BAA1BAL, 0x84EA524EL, 0x7681D14DL,
	0x2892ED69L, 0xDAF96E6AL, 0xC9A99D9EL, 0x3BC21E9DL,
	0xEF087A76L, 0x1D63F975L, 0x0E330A81L, 0xFC588982L,
	0xB21572C9L, 0x407EF1CAL, 0x532E023EL, 0xA145813DL,
	0x758FE5D6L, 0x87E466D5L, 0x94B49521L, 0x66DF1622L,
	0x38CC2A06L, 0xCAA7A905L, 0xD9F75AF1L, 0x2B9CD9F2L,
	0xFF56BD19L, 0x0D3D3E1AL, 0x1E6DCDEEL, 0xEC064EEDL,
	0xC38D26C4L, 0x31E6A5C7L, 0x22B65633L, 0xD0DDD530L,
	0x0417B1DBL, 0xF67C32D8L, 0xE52CC12CL, 0x1747422FL,
	0x49547E0BL, 0xBB3FFD08L, 0xA86F0EFCL, 0x5A048DFFL,
	0x8ECEE914L, 0x7CA56A17L, 0x6FF599E3L, 0x9D9E1AE0L,
	0xD3D3E1ABL, 0x21B862A8L, 0x32E8915CL, 0xC083125FL,
	0x144976B4L, 0xE622F5B7L, 0xF5720643L, 0x07198540L,
	0x590AB964L, 0xAB613A67L, 0xB831C993L, 0x4A5A4A90L,
	0x9E902E7BL, 0x6CFBAD78L, 0x7FAB5E8CL, 0x8DC0DD8FL,
	0xE330A81AL, 0x115B2B19L, 0x020BD8EDL, 0xF0605BEEL,
	0x24AA3F05L, 0xD6C1BC06L, 0xC5914FF2L, 0x37FACCF1L,
	0x69E9F0D5L, 0x9B8273D6L, 0x88D28022L, 0x7AB90321L,
	0xAE7367CAL, 0x5C18E4C9L, 0x4F48173DL, 0xBD23943EL,
	0xF36E6F75L, 0x0105EC76L, 0x12551F82L, 0xE03E9C81L,
	0x34F4F86AL, 0xC69F7B69L, 0xD5CF889DL, 0x27A40B9EL,
	0x79B737BAL, 0x8BDCB4B9L, 0x988C474DL, 0x6AE7C44EL,
	0xBE2DA0A5L, 0x4C4623A6L, 0x5F16D052L, 0xAD7D5351L
};
static uint32_t crc32( uint32_t sum, uint8_t next ) {
	uint8_t c = (uint8_t) sum ^ next;
	return pgm_read_dword( &crctable[c] ) ^ (sum >> 8);
}

static void setupMemoryAccess() {
	// Set up for reading memory
	DDRA = 0b11111111; // Address low
	DDRC = 0b11111111; // Address high
	DDRD = 0b00000000; // Data
	PORTD = 0b00000000;
	DDRH |= 0b00010111; // RD, WR, IORQ, MREQ
	PORTH = (PORTH | 0b00010111); // No mode
}

static void endMemoryAccess() {
	DDRA = 0b00000000; // Address low
	DDRC = 0b00000000; // Address high
	DDRD = 0b00000000; // Data
	PORTD = 0b00000000;
	DDRH &= 0b11101000; // Detach RD, WR, IORQ, MREQ
	PORTH &= 0b11101000;
}

static void readMode() {
	DDRD = 0b00000000; // Data
	PORTD = 0b00000000;
	PORTH = (PORTH & 0b11111101) | 0b00000001; // Set RD mode
}

static void writeMode() {
	DDRD = 0b11111111; // Data
	PORTH = (PORTH & 0b11111110) | 0b00000010; // Set WR mode
}

static uint8_t readMemory( uint16_t addr ) {
	PORTC = addr >> 8;
	PORTA = addr & 0xff;
	PORTH &= 0b11111011; // Set MREQ
	for( int i=0; i<10; i++) { _NOP(); }
	uint8_t ret = PIND;
	PORTH |= 0b00000100;

	return ret;
}

static void writeMemory( uint16_t addr, uint8_t data ) {
	PORTD = data;
	PORTC = addr >> 8;
	PORTA = addr & 0xff;
	PORTH &= 0b11111011; // Set MREQ
	for( int i=0; i<10; i++) { _NOP(); }
	PORTH |= 0b00000100;
}

static bool romChecksum() {
	if( !doBusAck() ) {
		return false;
	}
	printf_P( PSTR( "Testing ROM\n" ) );
	
	setupMemoryAccess();
	readMode();

	uint32_t sum = 0xffffffff;
	for( uint16_t addr = 0; addr < 0x2000; addr++ )
	{
		sum = crc32( sum, readMemory( addr ) );
	}
	sum ^= 0xffffffff;

	printf_P( PSTR( "ROM checksum: %08lx" ), (long unsigned int) sum );

	bool pass = true;
	switch( sum )
	{
		case 0x6605af34:
		printf_P( PSTR( " America OK\n" ) );
		break;
		case 0x0ff1a16f:
		printf_P( PSTR( " America (no-delay) OK\n" ) );
		break;
		case 0x4dd45e90:
		printf_P( PSTR( " America (alternate font) OK\n" ) );
		break;
		case 0x5c589f1b:
		printf_P( PSTR( " PAL/SECAM OK\n" ) );
		break;
		case 0xe7a9948c:
		printf_P( PSTR( " PAL/SECAM (NewColeco no-delay) OK\n" ) );
		break;
		default:
		printf_P( PSTR( " Fail\n" ) );
		pass = false;
	}
	endMemoryAccess();

	return pass;
}

static bool memoryTest() {
	if( !doBusAck() ) {
		return false;
	}
	printf_P( PSTR( "Testing system RAM\n" ) );
	setupMemoryAccess();

	uint8_t errors1 = 0;
    for( uint8_t i = 0; i < 21; i++ )
	{
		static tinymt32_t state;
		
		printf_P( PSTR( "Memory test pattern %u: " ), i );

		tinymt32_init( &state, i );
		
		writeMode();
		for( uint16_t addr = 0x7000; addr < 0x7400; addr++ )
		{
			writeMemory( addr, (uint8_t) tinymt32_generate_uint32( &state ) );
		}

		// Now read back
		tinymt32_init( &state, i );

		readMode();
		for( uint16_t addr = 0x7000; addr < 0x7400; addr++ )
		{
			uint8_t data = readMemory( addr ) ^ tinymt32_generate_uint32( &state );
			errors1 |= data;
		}

		if( errors1 != 0 )
		{
			printf_P( PSTR( "Errors encountered\n" ) );
			break;
		}
		
		printf_P( PSTR( "OK\n" ) );
	}

	endMemoryAccess();

	if( ( errors1 & 0xF0 ) != 0 ) {
		printf_P( PSTR( "U4 fail\n" ) );
	}
	if( ( errors1 & 0x0F ) != 0 ) {
		printf_P( PSTR( "U3 fail\n" ) );
	}

	if( errors1 == 0 ) {
		printf_P( PSTR( "RAM test passed\n" ) );
	}

	return errors1 == 0;
}


static bool cartridgeTest() {
	return true;
}

void doMemTests() {
	romChecksum();
	memoryTest() &&
	cartridgeTest();
}
